#include <afx.h>
#include <afxtempl.h>
#include "parse.h"
#include "topiclog.h"

#include "parsesrc.h"
#include "parsevb.h"
#include "resource.h"

const char *aszTokAs[] = { "As ", NULL, };
const char *aszTokProperty[] = { "Property ", NULL, };
const char *aszRoutineKeys[] = { "Function ", "Sub ", "Type ", NULL, };
const char *aszRoutineModKeys[] = { "Private ", "Public ", "Static ", "Friend ", NULL, };
const char *aszPropOps[] = { "Get ", "Let ", "Set ", NULL, };
const char *aszVarMods[] = { "Optional ", "ByVal ", "ByRef ", "ParamArray ", NULL, };
const char *szVbNameExt = "%$";

/*
@func Parses a VB field declaration. Allows parsing to resume at the beginning of the
argument list.

@parsedest 0 | Name
@parsedest 1 | Type name (optional)

@parsediag 

	  elementname [([subscripts])] As type

@parm Tag to populate.
@parm Text to parse. Resume parsing at first delimiter of parameter list.
*/

int ParseVbField(
	CTag *ptag,
	CParseText &txt)
{
	CSrcName    tName;
	CSrcParens  tParens;
	CSrcKeyName tType;

	CSrcItemList lVbField;
	CSrcTokenArray aSrcTokens;

	CString sTok;
	CString sT;

	tName.SetNameExtensions(szVbNameExt);

	tParens.SetDelims(chOpenParen, chCloseParen);
	tParens.SetOptional(TRUE);
	
	tType.SetKeyList(aszTokAs);
	tType.SetOptional(TRUE);

	lVbField.AddTail(&tName);
	lVbField.AddTail(&tParens);
	lVbField.AddTail(&tType);

	char *szCont;

	int nRet = LinkTokens(txt.m_szCur, szCont, lVbField, aSrcTokens, NULL);
	if(nRet)
		return nRet;

	nRet = CompleteTokens(lVbField, aSrcTokens);
	if(nRet)
		goto ExitError;

	ptag->ShiftFields(0, 2);

	// Copy the name
	
	tName.GetPhrase(sTok);
	if(tParens.GetPhrase(sT))
	{
		sTok += sT;
	}
	ptag->CopySrcString(0, sTok);

	// Copy the type (may be an empty)
	
	tType.GetToken(1, sTok);

	ptag->CopySrcString(1, sTok);
	
ExitError:
	ClearSrcTokens(aSrcTokens);

	return nRet;
}


		
/*
@func Parses VB parameter.

@parsedest 0 | Name modifiers 
@parsedest 1 | Parameter name
@parsedest 2 | Type name

@parsediag 

	{ ( | , } [Optional][ByVal | ByRef][ParamArray] varname[( )][As type]

@parm Tag to populate.
@parm Text to parse. There must be a paren or comma preceding the parameter
entry.
*/

int ParseVbParm(
	CTag *ptag,
	CParseText &txt)
{
	CSrcKeyList tMods;
	CSrcName    tName;
	CSrcParens  tParens;
	CSrcKeyName tType;

	CSrcItemList lVbField;
	CSrcTokenArray aSrcTokens;

	CString sTok;
	CString sT;

	txt.m_szCur = EatWhite(txt.m_szCur);
	if(chComma == *txt.m_szCur || chOpenParen == *txt.m_szCur)
		txt.m_szCur = EatWhite(txt.m_szCur + 1);
	else
		return warnSrcFieldsUnavailable;

	tName.SetNameExtensions(szVbNameExt);

	tMods.SetKeyList(aszVarMods);
	tMods.SetOptional(TRUE);
	
	tParens.SetDelims(chOpenParen, chCloseParen);
	tParens.SetOptional(TRUE);
	
	tType.SetKeyList(aszTokAs);
	tType.SetOptional(TRUE);

	lVbField.AddTail(&tMods);
	lVbField.AddTail(&tName);
	lVbField.AddTail(&tParens);
	lVbField.AddTail(&tType);

	char *szCont;

	int nRet = LinkTokens(txt.m_szCur, szCont, lVbField, aSrcTokens, NULL);
	if(nRet)
		return nRet;

	nRet = CompleteTokens(lVbField, aSrcTokens);
	if(nRet)
		goto ExitError;

	ptag->ShiftFields(0, 3);

	tMods.GetPhrase(sTok);
	ptag->CopySrcString(0, sTok);

	tName.GetPhrase(sTok);
	if(tParens.GetPhrase(sT))
	{
		sTok += sT;
	}
	ptag->CopySrcString(1, sTok);


	// Get the type (if it exists) and set the continuation
	// pointer to the appropriate location depending on whether
	// the type was parsed.

	CSrcToken *pSrcToken;

	if(tType.GetToken(1, sTok))
	{
		ptag->CopySrcString(2, sTok);
		tType.GetMatch(pSrcToken);
	}
	else
	{
		ptag->CopySrcString(2, NULL);
		tName.GetMatch(pSrcToken);
	}

	txt.m_szCur = pSrcToken->m_sz + pSrcToken->m_nLen;

ExitError:
	ClearSrcTokens(aSrcTokens);

	return nRet;
}



/*
@func Parses a VB subroutine, function, or type declaration, depending on
how the <p bCopyParens> and <b bCopyType> parameters are defined.

@parsedest 0 | Name modifiers 
@parsedest 1 | Subroutine name
@parsedest 2 | Parameter list (optionally copied if <p bCopyParm> is TRUE)
@parsedest 3 | Type name (optionally copies if <b pCopyType> is TRUE)

@parsediag 

[Public | Private][Static] Function name [(arglist)][As type]
	[statements]
	[name = expression]
	[Exit Function] 
	[statements]
	[name = expression]
End Function

[Private | Public][Static] Sub name [(arglist)] 
	[statements]
	[Exit Sub]
	[statements]
End Sub

[Private | Public] Type varname
	  elementname [([subscripts])] As type
	[ elementname [([subscripts])] As type]
	. . .
End Type

@parm Tag to populate.
@parm Text to parse. Resume parsing at first delimiter of parameter list.
@parm TRUE if we want to parse and insert an argument list
@parm TRUE if we want to parse and insert a type name.

*/

int ParseVbRoutine(CTag *ptag, CParseText &txt, BOOL bCopyParens, BOOL bCopyType)
{
	CSrcKeyList tMods;
	CSrcKeyName tName;
	CSrcParens  tParens;
	CSrcKeyName tType;

	CSrcItemList   lVbSub;
	CSrcTokenArray aSrcTokens;

	CString sTok;
	CString sT;

	int nShift;
	int nField;
	
	// Set up the source tokens.

	tMods.SetKeyList(aszRoutineModKeys);
	tMods.SetOptional(TRUE);

	tName.SetKeyList(aszRoutineKeys);
	tName.SetNameExtensions(szVbNameExt);

	tParens.SetDelims(chOpenParen, chCloseParen);
	tParens.SetOptional(TRUE);

	tType.SetOptional(TRUE);
	tType.SetKeyList(aszTokAs);

	lVbSub.AddTail(&tMods);
	lVbSub.AddTail(&tName);
	lVbSub.AddTail(&tParens);
	lVbSub.AddTail(&tType);

	char *szCont;

	int nRet = LinkTokens(txt.m_szCur, szCont, lVbSub, aSrcTokens, NULL);
	if(nRet)
		return nRet;

	nRet = CompleteTokens(lVbSub, aSrcTokens);
	if(nRet)
		goto ExitError;

	nShift = 2;

	if(bCopyParens)
		nShift += 1;

	if(bCopyType)
		nShift += 1;

	ptag->ShiftFields(0, nShift);

	// optional modifiers
	if(tMods.GetPhrase(sTok))
		ptag->CopySrcString(0, sTok);

	// required name
	tName.GetToken(1, sTok);
	ptag->CopySrcString(1, sTok);

	nField = 1;

	// optional (and maybe there if requested) parameter list -
	// if there's a parameter list need to get the start of it
	// so parsing can resume here.

	CSrcToken *pSrcToken;

	if(tParens.GetPhrase(sTok))
	{
		tParens.GetMatch(pSrcToken);
		txt.m_szCur = pSrcToken->m_sz;		
	}
	else
	{
		ptag->CopySrcString(nField, NULL);
		tName.GetMatch(pSrcToken);
		txt.m_szCur = pSrcToken->m_sz + pSrcToken->m_nLen;		
	}

	if(bCopyParens)
	{
		++nField;
		ptag->CopySrcString(nField, sTok);
	}

	// optional type
	if(bCopyType)
	{
		++nField;
		tType.GetToken(1, sTok);

		ptag->CopySrcString(nField, sTok);
	}

ExitError:
	ClearSrcTokens(aSrcTokens);

	return nRet;
}



/*
@func Parses a VB Property declaration. Allows parsing to resume at the beginning of the
argument list.

@parsedest 0 | Name modifiers
@parsedest 1 | Let/Get/Set Keyword
@parsedest 2 | Subroutine name
@parsedest 3 | Parameter list
@parsedest 4 | Type name

@parsediag 

[Public | Private | Friend] [Static] Property { Get | Let | Set } name [(arglist)] [As type]
	[statements]
	[name = expression]
	[Exit Property] 
	[statements]
	[name = expression]
End Property

@parm Tag to populate.
@parm Text to parse. Resume parsing at first delimiter of parameter list.
*/

int ParseVbProperty(CTag *ptag,	CParseText &txt)
{
	CSrcKeyList tMods;
	CSrcKey     tProperty;
	CSrcKeyName tName;
	CSrcParens  tParens;
	CSrcKeyName tType;

	CSrcItemList   lVbProp;
	CSrcTokenArray aSrcTokens;

	CString sTok;
	CString sT;

	char *szCont;

	// Set up the source tokens.

	tMods.SetKeyList(aszRoutineModKeys);
	tMods.SetOptional(TRUE);

	tProperty.SetKeyList(aszTokProperty);
	tProperty.SetOptional(FALSE);

	tName.SetKeyList(aszPropOps);
	tName.SetOptional(FALSE);

	tParens.SetDelims(chOpenParen, chCloseParen);
	tParens.SetOptional(TRUE);

	tType.SetOptional(TRUE);
	tType.SetKeyList(aszTokAs);

	lVbProp.AddTail(&tMods);
	lVbProp.AddTail(&tProperty);
	lVbProp.AddTail(&tName);
	lVbProp.AddTail(&tParens);
	lVbProp.AddTail(&tType);

	int nRet = LinkTokens(txt.m_szCur, szCont, lVbProp, aSrcTokens, NULL);
	if(nRet)
		return nRet;

	nRet = CompleteTokens(lVbProp, aSrcTokens);
	if(nRet)
		goto ExitError;

	ptag->ShiftFields(0, 5);

	// optional modifiers
	if(tMods.GetPhrase(sTok))
		ptag->CopySrcString(0, sTok);

	// required get/set/let keyword
	tName.GetToken(0, sTok);
	ptag->CopySrcString(1, sTok);

	// required name
	tName.GetToken(1, sTok);
	ptag->CopySrcString(2, sTok);

	// optional parameter list. if there is one, need to get the start of it
	// so parsing can resume here.

	CSrcToken *pSrcToken;

	if(tParens.GetPhrase(sTok))
	{
		ptag->CopySrcString(3, sTok);
		tParens.GetMatch(pSrcToken);
		txt.m_szCur = pSrcToken->m_sz + 1;		
	}
	else
	{
		ptag->CopySrcString(3, NULL);
		tName.GetMatch(pSrcToken);
		txt.m_szCur = pSrcToken->m_sz + pSrcToken->m_nLen;		
	}

	// optional type

	if(tType.GetToken(1, sTok))
		ptag->CopySrcString(4, sTok);
	else
		ptag->CopySrcString(4, NULL);

ExitError:
	ClearSrcTokens(aSrcTokens);

	return nRet;
}



